/*
 * This Software is licensed under one of the following licenses:
 *
 * 1) under the terms of the "Common Public License 1.0" a copy of which is
 *    available from the Open Source Initiative, see
 *    http://www.opensource.org/licenses/cpl.php.
 *
 * 2) under the terms of the "The BSD License" a copy of which is
 *    available from the Open Source Initiative, see
 *    http://www.opensource.org/licenses/bsd-license.php.
 *
 * 3) under the terms of the "GNU General Public License (GPL) Version 2" a
 *    copy of which is available from the Open Source Initiative, see
 *    http://www.opensource.org/licenses/gpl-license.php.
 *
 * Licensee has the right to choose one of the above licenses.
 *
 * Redistributions of source code must retain the above copyright
 * notice and one of the license notices.
 *
 * Redistributions in binary form must reproduce both the above copyright
 * notice, one of the license notices in the documentation
 * and/or other materials provided with the distribution.
 */

/***************************************************************************
 *
 *   Module:		 uDAPL
 *
 *   Filename:		 dapl_ib_cm.c
 *
 *   Author:		 Arlin Davis
 *
 *   Created:		 3/10/2005
 *
 *   Description: 
 *
 *   The uDAPL openib provider - connection management
 *
 ****************************************************************************
 *		   Source Control System Information
 *
 *    $Id: $
 *
 *	Copyright (c) 2005 Intel Corporation.  All rights reserved.
 *
 **************************************************************************/

#include "dapl.h"
#include "dapl_adapter_util.h"
#include "dapl_evd_util.h"
#include "dapl_cr_util.h"
#include "dapl_name_service.h"
#include "dapl_ib_util.h"

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <netinet/tcp.h>
#include <sysfs/libsysfs.h>

/* prototypes */
static uint16_t dapli_get_lid( struct ibv_device *dev, int port );

static DAT_RETURN dapli_socket_connect ( DAPL_EP		*ep_ptr,
					 DAT_IA_ADDRESS_PTR	r_addr,
					 DAT_CONN_QUAL		r_qual,
					 DAT_COUNT		p_size,
					 DAT_PVOID		p_data );

static DAT_RETURN dapli_socket_listen ( DAPL_IA			*ia_ptr,
					DAT_CONN_QUAL		serviceID,
					DAPL_SP			*sp_ptr );

static DAT_RETURN dapli_socket_accept(	ib_cm_srvc_handle_t cm_ptr );

static DAT_RETURN dapli_socket_accept_final(	DAPL_EP		*ep_ptr,
						DAPL_CR		*cr_ptr,
						DAT_COUNT	p_size,
						DAT_PVOID	p_data );

/* XXX temporary hack to get lid */
static uint16_t dapli_get_lid(IN struct ibv_device *dev, IN int port)
{
	char path[128];
	char val[16];
	char name[256];

	if (sysfs_get_mnt_path(path, sizeof path)) {
		fprintf(stderr, "Couldn't find sysfs mount.\n");
		return 0;
	}
	sprintf(name, "%s/class/infiniband/%s/ports/%d/lid", path,
		 ibv_get_device_name(dev), port);

	if (sysfs_read_attribute_value(name, val, sizeof val)) {
		fprintf(stderr, "Couldn't read LID at %s\n", name);
		return 0;
	}
	return strtol(val, NULL, 0);
}

/*
 * ACTIVE: Create socket, connect, and exchange QP information 
 */
static DAT_RETURN 
dapli_socket_connect (	DAPL_EP			*ep_ptr,
			DAT_IA_ADDRESS_PTR	r_addr,
			DAT_CONN_QUAL		r_qual,
			DAT_COUNT		p_size,
			DAT_PVOID		p_data )
{
	ib_cm_handle_t	cm_ptr;
	DAPL_IA		*ia_ptr = ep_ptr->header.owner_ia;
	int		len, opt = 1;
	struct iovec    iovec[2];
	short		rtu_data = htons(0x0E0F);
	
	dapl_dbg_log(DAPL_DBG_TYPE_EP, " connect: r_qual %d\n", r_qual);
			
	/*
	 *  Allocate CM and initialize
	 */
	if ((cm_ptr = dapl_os_alloc(sizeof(*cm_ptr))) == NULL ) {
		return DAT_INSUFFICIENT_RESOURCES;
	}

	(void) dapl_os_memzero( cm_ptr, sizeof( *cm_ptr ) );
	cm_ptr->socket = -1;

	/* create, connect, sockopt, and exchange QP information */
	if ((cm_ptr->socket = socket(AF_INET,SOCK_STREAM,0)) < 0 ) {
		dapl_os_free( cm_ptr, sizeof( *cm_ptr ) );
		return DAT_INSUFFICIENT_RESOURCES;
	}

	((struct sockaddr_in*)r_addr)->sin_port = htons(r_qual);

	if ( connect(cm_ptr->socket, r_addr, sizeof(*r_addr)) < 0 ) {
		dapl_dbg_log(DAPL_DBG_TYPE_ERR,
			     " connect: %s on r_qual %d\n",
			     strerror(errno), (unsigned int)r_qual);
		dapl_os_free( cm_ptr, sizeof( *cm_ptr ) );
		return DAT_INVALID_ADDRESS;
	}
	setsockopt(cm_ptr->socket,IPPROTO_TCP,TCP_NODELAY,&opt,sizeof(opt));
	
	/* Send QP info, IA address, and private data */
	cm_ptr->dst.qpn = ep_ptr->qp_handle->qp_num;
	cm_ptr->dst.port = ia_ptr->hca_ptr->port_num;
	cm_ptr->dst.lid = dapli_get_lid( ia_ptr->hca_ptr->ib_trans.ib_dev, 
					 ia_ptr->hca_ptr->port_num );
	cm_ptr->dst.ia_address = ia_ptr->hca_ptr->hca_address;
	cm_ptr->dst.p_size = p_size;
	iovec[0].iov_base = &cm_ptr->dst;
	iovec[0].iov_len  = sizeof(ib_qp_cm_t);
	if ( p_size ) {
		iovec[1].iov_base = p_data;
		iovec[1].iov_len  = p_size;
	}
	len = writev( cm_ptr->socket, iovec, (p_size ? 2:1) );
    	if ( len != (p_size + sizeof(ib_qp_cm_t)) ) {
		dapl_dbg_log(DAPL_DBG_TYPE_ERR, 
			     " connect write: ERR %s, wcnt=%d\n",
			     strerror(errno), len); 
		goto bail;
	}
	dapl_dbg_log(DAPL_DBG_TYPE_EP, 
		     " connect: SRC port=0x%x lid=0x%x, qpn=0x%x, psize=%d\n",
		     cm_ptr->dst.port, cm_ptr->dst.lid, 
		     cm_ptr->dst.qpn, cm_ptr->dst.p_size ); 

	/* read DST information into cm_ptr, overwrite SRC info */
	len = readv( cm_ptr->socket, iovec, 1 );
	if ( len != sizeof(ib_qp_cm_t) ) {
		dapl_dbg_log(DAPL_DBG_TYPE_ERR, 
			     " connect read: ERR %s, rcnt=%d\n",
			     strerror(errno), len); 
		goto bail;
	}
	dapl_dbg_log(DAPL_DBG_TYPE_EP, 
		     " connect: DST port=0x%x lid=0x%x, qpn=0x%x, psize=%d\n",
		     cm_ptr->dst.port, cm_ptr->dst.lid, 
		     cm_ptr->dst.qpn, cm_ptr->dst.p_size ); 

	/* validate private data size before reading */
	if ( cm_ptr->dst.p_size > IB_MAX_REP_PDATA_SIZE ) {
		dapl_dbg_log(DAPL_DBG_TYPE_ERR, 
			     " connect read: psize (%d) wrong\n",
			     cm_ptr->dst.p_size ); 
		goto bail;
	}

	/* read private data into cm_handle if any present */
	if ( cm_ptr->dst.p_size ) {
		iovec[0].iov_base = cm_ptr->p_data;
		iovec[0].iov_len  = cm_ptr->dst.p_size;
		len = readv( cm_ptr->socket, iovec, 1 );
		if ( len != cm_ptr->dst.p_size ) {
			dapl_dbg_log(DAPL_DBG_TYPE_ERR, 
				" connect read pdata: ERR %s, rcnt=%d\n",
				strerror(errno), len); 
			goto bail;
		}
	}

	/* modify QP to RTR and then to RTS with remote info */
	if ( dapls_modify_qp_state( ep_ptr->qp_handle, 
				    IBV_QPS_RTR, &cm_ptr->dst ) != DAT_SUCCESS )
		goto bail;

	if ( dapls_modify_qp_state( ep_ptr->qp_handle, 
				    IBV_QPS_RTS, &cm_ptr->dst ) != DAT_SUCCESS )
		goto bail;
		 
	ep_ptr->qp_state = IB_QP_STATE_RTS;

	/* complete handshake after final QP state change */
	write(cm_ptr->socket, &rtu_data, sizeof(rtu_data) );

	/* init cm_handle and post the event with private data */
	ep_ptr->cm_handle = cm_ptr;
	dapl_dbg_log( DAPL_DBG_TYPE_EP," ACTIVE: connected!\n" ); 
	dapl_evd_connection_callback(   ep_ptr->cm_handle, 
					IB_CME_CONNECTED, 
					cm_ptr->p_data, 
					ep_ptr );	
	return DAT_SUCCESS;

bail:
	/* close socket, free cm structure and post error event */
	if ( cm_ptr->socket >= 0 ) 
		close(cm_ptr->socket);
	dapl_os_free( cm_ptr, sizeof( *cm_ptr ) );
	dapls_ib_reinit_ep( ep_ptr ); /* reset QP state */

	dapl_evd_connection_callback(	ep_ptr->cm_handle, 
					IB_CME_LOCAL_FAILURE, 
					NULL, 
					ep_ptr );
	return DAT_INTERNAL_ERROR;
}


/*
 * PASSIVE: Create socket, listen, accept, exchange QP information 
 */
static DAT_RETURN 
dapli_socket_listen (	DAPL_IA		*ia_ptr,
			DAT_CONN_QUAL	serviceID,
			DAPL_SP		*sp_ptr )
{
	struct sockaddr_in	addr;
	ib_cm_srvc_handle_t	cm_ptr = NULL;
	int			opt = 1;
	DAT_RETURN		dat_status = DAT_SUCCESS;

	dapl_dbg_log (	DAPL_DBG_TYPE_EP,
			" listen(ia_ptr %p ServiceID %d sp_ptr %p)\n",
			ia_ptr, serviceID, sp_ptr);

	/* Allocate CM and initialize */
	if ((cm_ptr = dapl_os_alloc(sizeof(*cm_ptr))) == NULL) 
		return DAT_INSUFFICIENT_RESOURCES;

	(void) dapl_os_memzero( cm_ptr, sizeof( *cm_ptr ) );
	
	cm_ptr->socket = cm_ptr->l_socket = -1;
	cm_ptr->sp = sp_ptr;
	cm_ptr->hca_ptr = ia_ptr->hca_ptr;
	
	/* bind, listen, set sockopt, accept, exchange data */
	if ((cm_ptr->l_socket = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		dapl_dbg_log (DAPL_DBG_TYPE_ERR, 
				"socket for listen returned %d\n", errno);
		dat_status = DAT_INSUFFICIENT_RESOURCES;
		goto bail;
	}

	setsockopt(cm_ptr->l_socket,SOL_SOCKET,SO_REUSEADDR,&opt,sizeof(opt));
	addr.sin_port        = htons(serviceID);
	addr.sin_family      = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;

	if (( bind( cm_ptr->l_socket,(struct sockaddr*)&addr, sizeof(addr) ) < 0) ||
		   (listen( cm_ptr->l_socket, 128 ) < 0) ) {
	
		dapl_dbg_log( DAPL_DBG_TYPE_CM,
				" listen: ERROR %s on conn_qual 0x%x\n",
				strerror(errno),serviceID); 

		if ( errno == EADDRINUSE )
			dat_status = DAT_CONN_QUAL_IN_USE;
		else
			dat_status = DAT_CONN_QUAL_UNAVAILABLE;

		goto bail;
	}
	
	/* set cm_handle for this service point, save listen socket */
	sp_ptr->cm_srvc_handle = cm_ptr;

	/* add to SP->CR thread list */
	dapl_llist_init_entry((DAPL_LLIST_ENTRY*)&cm_ptr->entry);
	dapl_os_lock( &cm_ptr->hca_ptr->ib_trans.lock );
	dapl_llist_add_tail(&cm_ptr->hca_ptr->ib_trans.list, 
			    (DAPL_LLIST_ENTRY*)&cm_ptr->entry, cm_ptr);
	dapl_os_unlock(&cm_ptr->hca_ptr->ib_trans.lock);

	dapl_dbg_log( DAPL_DBG_TYPE_CM,
			" listen: qual 0x%x cr %p s_fd %d\n",
			ntohs(serviceID), cm_ptr, cm_ptr->l_socket ); 
	
	return dat_status;
bail:
	dapl_dbg_log( DAPL_DBG_TYPE_CM,
			" listen: ERROR on conn_qual 0x%x\n",serviceID); 
	if ( cm_ptr->l_socket >= 0 )
		close( cm_ptr->l_socket );
	dapl_os_free( cm_ptr, sizeof( *cm_ptr ) );
	return dat_status;
}


/*
 * PASSIVE: send local QP information, private data, and wait for 
 *	    active side to respond with QP RTS/RTR status 
 */
static DAT_RETURN 
dapli_socket_accept(ib_cm_srvc_handle_t cm_ptr)
{
	ib_cm_handle_t	acm_ptr;
	void		*p_data = NULL;
	int		len;
	DAT_RETURN	dat_status = DAT_SUCCESS;
		
	/* Allocate accept CM and initialize */
	if ((acm_ptr = dapl_os_alloc(sizeof(*acm_ptr))) == NULL) 
		return DAT_INSUFFICIENT_RESOURCES;

	(void) dapl_os_memzero( acm_ptr, sizeof( *acm_ptr ) );
	
	acm_ptr->socket = -1;
	acm_ptr->sp = cm_ptr->sp;
	acm_ptr->hca_ptr = cm_ptr->hca_ptr;

	len = sizeof(acm_ptr->dst.ia_address);
	acm_ptr->socket = accept(cm_ptr->l_socket, 
				(struct sockaddr*)&acm_ptr->dst.ia_address, 
				&len );

	if ( acm_ptr->socket < 0 ) {
		dapl_dbg_log(DAPL_DBG_TYPE_ERR, 
			" accept: ERR %s on FD %d l_cr %p\n",
			strerror(errno),cm_ptr->l_socket,cm_ptr); 
		dat_status = DAT_INTERNAL_ERROR;
		goto bail;
   	}

	/* read in DST QP info, IA address. check for private data */
	len = read( acm_ptr->socket, &acm_ptr->dst, sizeof(ib_qp_cm_t) );
	if ( len != sizeof(ib_qp_cm_t) ) {
		dapl_dbg_log(DAPL_DBG_TYPE_ERR, 
			" accept read: ERR %s, rcnt=%d\n",
			strerror(errno), len); 
		dat_status = DAT_INTERNAL_ERROR;
		goto bail;

	}
	dapl_dbg_log(DAPL_DBG_TYPE_EP, 
		" accept: DST port=0x%x lid=0x%x, qpn=0x%x, psize=%d\n",
		acm_ptr->dst.port, acm_ptr->dst.lid, 
		acm_ptr->dst.qpn, acm_ptr->dst.p_size ); 

	/* validate private data size before reading */
	if ( acm_ptr->dst.p_size > IB_MAX_REQ_PDATA_SIZE ) {
		dapl_dbg_log(DAPL_DBG_TYPE_ERR, 
			" accept read: psize (%d) wrong\n",
			acm_ptr->dst.p_size ); 
		dat_status = DAT_INTERNAL_ERROR;
		goto bail;
	}

	/* read private data into cm_handle if any present */
	if ( acm_ptr->dst.p_size ) {
		len = read( acm_ptr->socket, 
			    acm_ptr->p_data, acm_ptr->dst.p_size );
		if ( len != acm_ptr->dst.p_size ) {
			dapl_dbg_log(DAPL_DBG_TYPE_ERR, 
				" accept read pdata: ERR %s, rcnt=%d\n",
				strerror(errno), len ); 
			dat_status = DAT_INTERNAL_ERROR;
			goto bail;
		}
		dapl_dbg_log(DAPL_DBG_TYPE_EP, 
				" accept: psize=%d read\n",
				acm_ptr->dst.p_size); 
		p_data = acm_ptr->p_data;
	}
	
	/* trigger CR event and return SUCCESS */
	dapls_cr_callback(  acm_ptr,
			    IB_CME_CONNECTION_REQUEST_PENDING,
		            p_data,
			    acm_ptr->sp );

	return DAT_SUCCESS;

bail:
	if ( acm_ptr->socket >=0 )
		close( acm_ptr->socket );
	dapl_os_free( acm_ptr, sizeof( *acm_ptr ) );
	return DAT_INTERNAL_ERROR;
}


static DAT_RETURN 
dapli_socket_accept_final( DAPL_EP		*ep_ptr,
			   DAPL_CR		*cr_ptr,
			   DAT_COUNT		p_size,
		           DAT_PVOID		p_data )
{
	DAPL_IA		*ia_ptr = ep_ptr->header.owner_ia;
	ib_cm_handle_t	cm_ptr = cr_ptr->ib_cm_handle;
	ib_qp_cm_t	qp_cm;
	struct iovec    iovec[2];
	int		len;
	short		rtu_data = 0;

	if (p_size >  IB_MAX_REP_PDATA_SIZE) 
		return DAT_LENGTH_ERROR;

	/* must have a accepted socket */
	if ( cm_ptr->socket < 0 )
		return DAT_INTERNAL_ERROR;
	
	/* modify QP to RTR and then to RTS with remote info already read */
	if ( dapls_modify_qp_state( ep_ptr->qp_handle, 
				    IBV_QPS_RTR, &cm_ptr->dst ) != DAT_SUCCESS )
		goto bail;

	if ( dapls_modify_qp_state( ep_ptr->qp_handle, 
				    IBV_QPS_RTS, &cm_ptr->dst ) != DAT_SUCCESS )
		goto bail;

	ep_ptr->qp_state = IB_QP_STATE_RTS;
	
	/* Send QP info, IA address, and private data */
	qp_cm.qpn = ep_ptr->qp_handle->qp_num;
	qp_cm.port = ia_ptr->hca_ptr->port_num;
	qp_cm.lid = dapli_get_lid( ia_ptr->hca_ptr->ib_trans.ib_dev, 
				   ia_ptr->hca_ptr->port_num );
	qp_cm.ia_address = ia_ptr->hca_ptr->hca_address;
	qp_cm.p_size = p_size;
	iovec[0].iov_base = &qp_cm;
	iovec[0].iov_len  = sizeof(ib_qp_cm_t);
	if (p_size) {
		iovec[1].iov_base = p_data;
		iovec[1].iov_len  = p_size;
	}
	len = writev( cm_ptr->socket, iovec, (p_size ? 2:1) );
    	if (len != (p_size + sizeof(ib_qp_cm_t))) {
		dapl_dbg_log(DAPL_DBG_TYPE_ERR, 
			     " accept_final: ERR %s, wcnt=%d\n",
			     strerror(errno), len); 
		goto bail;
	}
	dapl_dbg_log(DAPL_DBG_TYPE_EP, 
		     " accept_final: SRC port=0x%x lid=0x%x, qpn=0x%x, psize=%d\n",
		     qp_cm.port, qp_cm.lid, qp_cm.qpn, qp_cm.p_size ); 
	
	/* complete handshake after final QP state change */
	len = read(cm_ptr->socket, &rtu_data, sizeof(rtu_data) );
	if ( len != sizeof(rtu_data) || ntohs(rtu_data) != 0x0e0f ) {
		dapl_dbg_log(DAPL_DBG_TYPE_ERR, 
			     " accept_final: ERR %s, rcnt=%d rdata=%x\n",
			     strerror(errno), len, ntohs(rtu_data) ); 
		goto bail;
	}

	/* final data exchange if remote QP state is good to go */
	dapl_dbg_log( DAPL_DBG_TYPE_EP," PASSIVE: connected!\n" ); 
	dapls_cr_callback ( cm_ptr, IB_CME_CONNECTED, NULL, cm_ptr->sp );
	return DAT_SUCCESS;

bail:
	dapl_dbg_log( DAPL_DBG_TYPE_ERR," accept_final: ERR !QP_RTR_RTS \n"); 
	if ( cm_ptr >= 0 )
		close( cm_ptr->socket );
	dapl_os_free( cm_ptr, sizeof( *cm_ptr ) );
	dapls_ib_reinit_ep( ep_ptr ); /* reset QP state */

	return DAT_INTERNAL_ERROR;
}


/*
 * dapls_ib_connect
 *
 * Initiate a connection with the passive listener on another node
 *
 * Input:
 *	ep_handle,
 *	remote_ia_address,
 *	remote_conn_qual,
 *	prd_size		size of private data and structure
 *	prd_prt			pointer to private data structure
 *
 * Output:
 * 	none
 *
 * Returns:
 * 	DAT_SUCCESS
 *	DAT_INSUFFICIENT_RESOURCES
 *	DAT_INVALID_PARAMETER
 *
 */
DAT_RETURN
dapls_ib_connect (
	IN  DAT_EP_HANDLE		ep_handle,
	IN  DAT_IA_ADDRESS_PTR		remote_ia_address,
	IN  DAT_CONN_QUAL		remote_conn_qual,
	IN  DAT_COUNT			private_data_size,
	IN  void			*private_data )
{
	DAPL_EP		*ep_ptr;
	ib_qp_handle_t	qp_ptr;
	
	dapl_dbg_log ( DAPL_DBG_TYPE_EP,
			" connect(ep_handle %p ....)\n", ep_handle);
	/*
	 *  Sanity check
	 */
	if ( NULL == ep_handle ) 
		return DAT_SUCCESS;

	ep_ptr = (DAPL_EP*)ep_handle;
	qp_ptr = ep_ptr->qp_handle;

	return (dapli_socket_connect(	ep_ptr, remote_ia_address, 
					remote_conn_qual,
					private_data_size, private_data ));
}

/*
 * dapls_ib_disconnect
 *
 * Disconnect an EP
 *
 * Input:
 *	ep_handle,
 *	disconnect_flags
 *
 * Output:
 * 	none
 *
 * Returns:
 * 	DAT_SUCCESS
 *
 */
DAT_RETURN
dapls_ib_disconnect (
	IN	DAPL_EP			*ep_ptr,
	IN	DAT_CLOSE_FLAGS		close_flags )
{
	ib_cm_handle_t	cm_ptr = ep_ptr->cm_handle;

	dapl_dbg_log (DAPL_DBG_TYPE_EP,
			"dapls_ib_disconnect(ep_handle %p ....)\n",
			ep_ptr);

	if ( cm_ptr->socket >= 0 ) {
		close( cm_ptr->socket );
		cm_ptr->socket = -1;
	}
	
	/* reinit to modify QP state */
	dapls_ib_reinit_ep(ep_ptr);

	if ( ep_ptr->cr_ptr ) {
		dapls_cr_callback ( ep_ptr->cm_handle,
				    IB_CME_DISCONNECTED,
				    NULL,
				    ((DAPL_CR *)ep_ptr->cr_ptr)->sp_ptr );
	} else {
		dapl_evd_connection_callback ( ep_ptr->cm_handle,
						IB_CME_DISCONNECTED,
						NULL,
						ep_ptr );
		ep_ptr->cm_handle = NULL;
		dapl_os_free( cm_ptr, sizeof( *cm_ptr ) );
	}	
	return DAT_SUCCESS;
}

/*
 * dapls_ib_disconnect_clean
 *
 * Clean up outstanding connection data. This routine is invoked
 * after the final disconnect callback has occurred. Only on the
 * ACTIVE side of a connection.
 *
 * Input:
 *	ep_ptr		DAPL_EP
 *	active		Indicates active side of connection
 *
 * Output:
 * 	none
 *
 * Returns:
 * 	void
 *
 */
void
dapls_ib_disconnect_clean (
	IN  DAPL_EP			*ep_ptr,
	IN  DAT_BOOLEAN			active,
	IN  const ib_cm_events_t	ib_cm_event )
{
    return;
}

/*
 * dapl_ib_setup_conn_listener
 *
 * Have the CM set up a connection listener.
 *
 * Input:
 *	ibm_hca_handle		HCA handle
 *	qp_handle			QP handle
 *
 * Output:
 * 	none
 *
 * Returns:
 * 	DAT_SUCCESS
 *	DAT_INSUFFICIENT_RESOURCES
 *	DAT_INTERNAL_ERROR
 *	DAT_CONN_QUAL_UNAVAILBLE
 *	DAT_CONN_QUAL_IN_USE
 *
 */
DAT_RETURN
dapls_ib_setup_conn_listener (
	IN  DAPL_IA		*ia_ptr,
	IN  DAT_UINT64		ServiceID,
	IN  DAPL_SP		*sp_ptr )
{
	return (dapli_socket_listen( ia_ptr, ServiceID, sp_ptr ));
}


/*
 * dapl_ib_remove_conn_listener
 *
 * Have the CM remove a connection listener.
 *
 * Input:
 *	ia_handle		IA handle
 *	ServiceID		IB Channel Service ID
 *
 * Output:
 * 	none
 *
 * Returns:
 * 	DAT_SUCCESS
 *	DAT_INVALID_STATE
 *
 */
DAT_RETURN
dapls_ib_remove_conn_listener (
	IN  DAPL_IA		*ia_ptr,
	IN  DAPL_SP		*sp_ptr )
{
	ib_cm_srvc_handle_t	cm_ptr = sp_ptr->cm_srvc_handle;

	dapl_dbg_log (DAPL_DBG_TYPE_EP,
			"dapls_ib_remove_conn_listener(ia_ptr %p sp_ptr %p cm_ptr %p)\n",
			ia_ptr, sp_ptr, cm_ptr );

	/* close accepted socket, free cm_srvc_handle and return */
	if ( cm_ptr != NULL ) {
		if ( cm_ptr->l_socket >= 0 ) {
			close( cm_ptr->l_socket );
			cm_ptr->socket = -1;
		}
	    	/* cr_thread will free */
		sp_ptr->cm_srvc_handle = NULL;
	}
	return DAT_SUCCESS;
}

/*
 * dapls_ib_accept_connection
 *
 * Perform necessary steps to accept a connection
 *
 * Input:
 *	cr_handle
 *	ep_handle
 *	private_data_size
 *	private_data
 *
 * Output:
 * 	none
 *
 * Returns:
 * 	DAT_SUCCESS
 *	DAT_INSUFFICIENT_RESOURCES
 *	DAT_INTERNAL_ERROR
 *
 */
DAT_RETURN
dapls_ib_accept_connection (
	IN  DAT_CR_HANDLE	cr_handle,
	IN  DAT_EP_HANDLE	ep_handle,
	IN  DAT_COUNT		p_size,
	IN  const DAT_PVOID	p_data )
{
	DAPL_CR			*cr_ptr;
	DAPL_EP			*ep_ptr;
	
	dapl_dbg_log (DAPL_DBG_TYPE_EP,
		      "dapls_ib_accept_connection(cr %p ep %p prd %p,%d)\n",
		      cr_handle, ep_handle, p_data, p_size  );

	cr_ptr  = (DAPL_CR *) cr_handle;
	ep_ptr  = (DAPL_EP *) ep_handle;
	
	/* allocate and attach a QP if necessary */
	if ( ep_ptr->qp_state == DAPL_QP_STATE_UNATTACHED ) {
		DAT_RETURN status;
		status = dapls_ib_qp_alloc( ep_ptr->header.owner_ia, 
					    ep_ptr, ep_ptr );
		if ( status != DAT_SUCCESS )
    			return status;
	}
    
	return ( dapli_socket_accept_final(ep_ptr, cr_ptr, p_size, p_data) );
}


/*
 * dapls_ib_reject_connection
 *
 * Reject a connection
 *
 * Input:
 *	cr_handle
 *
 * Output:
 * 	none
 *
 * Returns:
 * 	DAT_SUCCESS
 *	DAT_INTERNAL_ERROR
 *
 */
DAT_RETURN
dapls_ib_reject_connection (
	IN  ib_cm_handle_t	ib_cm_handle,
	IN  int			reject_reason )
{
    	ib_cm_srvc_handle_t	cm_ptr = ib_cm_handle;

	dapl_dbg_log (DAPL_DBG_TYPE_EP,
		      "dapls_ib_reject_connection(cm_handle %p reason %x)\n",
		      ib_cm_handle, reject_reason );

	/* just close the socket and return */
	if ( cm_ptr->socket > 0 ) {
		close( cm_ptr->socket );
		cm_ptr->socket = -1;
	}
	return DAT_SUCCESS;
}

/*
 * dapls_ib_cm_remote_addr
 *
 * Obtain the remote IP address given a connection
 *
 * Input:
 *	cr_handle
 *
 * Output:
 *	remote_ia_address: where to place the remote address
 *
 * Returns:
 * 	DAT_SUCCESS
 *	DAT_INVALID_HANDLE
 *
 */
DAT_RETURN
dapls_ib_cm_remote_addr (
	IN      DAT_HANDLE	dat_handle,
	OUT	DAT_SOCK_ADDR6	*remote_ia_address )
{
	DAPL_HEADER	*header;
	ib_cm_handle_t	ib_cm_handle;

	dapl_dbg_log (DAPL_DBG_TYPE_EP,
		      "dapls_ib_cm_remote_addr(dat_handle %p, ....)\n",
		      dat_handle );

	header = (DAPL_HEADER *)dat_handle;

	if (header->magic == DAPL_MAGIC_EP) 
		ib_cm_handle = ((DAPL_EP *) dat_handle)->cm_handle;
	else if (header->magic == DAPL_MAGIC_CR) 
		ib_cm_handle = ((DAPL_CR *) dat_handle)->ib_cm_handle;
	else 
		return DAT_INVALID_HANDLE;

	dapl_os_memcpy(	remote_ia_address, 
			&ib_cm_handle->dst.ia_address, 
			sizeof(DAT_SOCK_ADDR6) );

	return DAT_SUCCESS;
}

/*
 * dapls_ib_private_data_size
 *
 * Return the size of private data given a connection op type
 *
 * Input:
 *	prd_ptr		private data pointer
 *	conn_op		connection operation type
 *
 * If prd_ptr is NULL, this is a query for the max size supported by
 * the provider, otherwise it is the actual size of the private data
 * contained in prd_ptr.
 *
 *
 * Output:
 *	None
 *
 * Returns:
 * 	length of private data
 *
 */
int dapls_ib_private_data_size (
	IN      DAPL_PRIVATE	*prd_ptr,
	IN	DAPL_PDATA_OP	conn_op)
{
	int  size;

	switch (conn_op)
	{
		case DAPL_PDATA_CONN_REQ:
		{
			size = IB_MAX_REQ_PDATA_SIZE;
			break;
		}
		case DAPL_PDATA_CONN_REP:
		{
			size = IB_MAX_REP_PDATA_SIZE;
			break;
		}
		case DAPL_PDATA_CONN_REJ:
		{
			size = IB_MAX_REJ_PDATA_SIZE;
			break;
		}
		case DAPL_PDATA_CONN_DREQ:
		{
			size = IB_MAX_DREQ_PDATA_SIZE;
			break;
		}
		case DAPL_PDATA_CONN_DREP:
		{
			size = IB_MAX_DREP_PDATA_SIZE;
			break;
		}
		default:
		{
			size = 0;
		}

	} /* end case */

	return size;
}

/*
 * Map all socket CM event codes to the DAT equivelent.
 */
#define DAPL_IB_EVENT_CNT	11

static struct ib_cm_event_map
{
	const ib_cm_events_t	ib_cm_event;
	DAT_EVENT_NUMBER	dat_event_num;
	} ib_cm_event_map[DAPL_IB_EVENT_CNT] = {
	/* 00 */  { IB_CME_CONNECTED,	
					DAT_CONNECTION_EVENT_ESTABLISHED}, 
	/* 01 */  { IB_CME_DISCONNECTED,	
					DAT_CONNECTION_EVENT_DISCONNECTED},
	/* 02 */  { IB_CME_DISCONNECTED_ON_LINK_DOWN, 
					DAT_CONNECTION_EVENT_DISCONNECTED},
	/* 03 */  { IB_CME_CONNECTION_REQUEST_PENDING,	
					DAT_CONNECTION_REQUEST_EVENT},
	/* 04 */  { IB_CME_CONNECTION_REQUEST_PENDING_PRIVATE_DATA,
					DAT_CONNECTION_REQUEST_EVENT},
	/* 05 */  { IB_CME_DESTINATION_REJECT,
					DAT_CONNECTION_EVENT_NON_PEER_REJECTED},
	/* 06 */  { IB_CME_DESTINATION_REJECT_PRIVATE_DATA,		
					DAT_CONNECTION_EVENT_PEER_REJECTED},
	/* 07 */  { IB_CME_DESTINATION_UNREACHABLE,	
					DAT_CONNECTION_EVENT_UNREACHABLE},
	/* 08 */  { IB_CME_TOO_MANY_CONNECTION_REQUESTS,
					DAT_CONNECTION_EVENT_NON_PEER_REJECTED},
	/* 09 */  { IB_CME_LOCAL_FAILURE,
					DAT_CONNECTION_EVENT_BROKEN},
	/* 10 */  { IB_CM_LOCAL_FAILURE,
					DAT_CONNECTION_EVENT_BROKEN}
};
 
/*
 * dapls_ib_get_cm_event
 *
 * Return a DAT connection event given a provider CM event.
 *
 * Input:
 *	dat_event_num	DAT event we need an equivelent CM event for
 *
 * Output:
 * 	none
 *
 * Returns:
 * 	ib_cm_event of translated DAPL value
 */
DAT_EVENT_NUMBER
dapls_ib_get_dat_event (
	IN    const ib_cm_events_t	ib_cm_event,
	IN    DAT_BOOLEAN		active)
{
	DAT_EVENT_NUMBER	dat_event_num;
	int			i;
	
	active = active;

	if (ib_cm_event > IB_CM_LOCAL_FAILURE)
		return (DAT_EVENT_NUMBER) 0;

	dat_event_num = 0;
	for (i = 0; i < DAPL_IB_EVENT_CNT; i++) {
		if (ib_cm_event == ib_cm_event_map[i].ib_cm_event) {
			dat_event_num = ib_cm_event_map[i].dat_event_num;
			break;
		}
	}
	dapl_dbg_log (DAPL_DBG_TYPE_CALLBACK,
		"dapls_ib_get_dat_event: event translate(%s) ib=0x%x dat=0x%x\n",
		active ? "active" : "passive",  ib_cm_event, dat_event_num);

	return dat_event_num;
}


/*
 * dapls_ib_get_dat_event
 *
 * Return a DAT connection event given a provider CM event.
 * 
 * Input:
 *	ib_cm_event	event provided to the dapl callback routine
 *	active		switch indicating active or passive connection
 *
 * Output:
 * 	none
 *
 * Returns:
 * 	DAT_EVENT_NUMBER of translated provider value
 */
ib_cm_events_t
dapls_ib_get_cm_event (
	IN    DAT_EVENT_NUMBER		dat_event_num)
{
    ib_cm_events_t	ib_cm_event;
    int			i;

    ib_cm_event = 0;
    for (i = 0; i < DAPL_IB_EVENT_CNT; i++) {
	if ( dat_event_num == ib_cm_event_map[i].dat_event_num ) {
		ib_cm_event = ib_cm_event_map[i].ib_cm_event;
		break;
	}
    }
    return ib_cm_event;
}

/* async CR processing thread to avoid blocking applications */
void cr_thread(void *arg) 
{
    struct dapl_hca	*hca_ptr = arg;
    ib_cm_srvc_handle_t	cr, next_cr;
    int			max_fd;
    fd_set		rfd,rfds;
    struct timeval	to;
     
    dapl_dbg_log(DAPL_DBG_TYPE_UTIL," cr_thread: ENTER hca %p\n",hca_ptr);

    dapl_os_lock( &hca_ptr->ib_trans.lock );
    hca_ptr->ib_trans.cr_state = IB_THREAD_RUN;
    while (hca_ptr->ib_trans.cr_state == IB_THREAD_RUN) {
	
	FD_ZERO( &rfds ); 
	max_fd = -1;
	
	if (!dapl_llist_is_empty(&hca_ptr->ib_trans.list))
            next_cr = dapl_llist_peek_head (&hca_ptr->ib_trans.list);
	else
	    next_cr = NULL;

	while (next_cr) {
	    cr = next_cr;
	    dapl_dbg_log (DAPL_DBG_TYPE_CM," thread: cm_ptr %p\n", cr );
	    if (cr->l_socket == -1 || 
		hca_ptr->ib_trans.cr_state != IB_THREAD_RUN) {

		dapl_dbg_log(DAPL_DBG_TYPE_CM," thread: Freeing %p\n", cr);
		next_cr = dapl_llist_next_entry(&hca_ptr->ib_trans.list,
						(DAPL_LLIST_ENTRY*)&cr->entry );
		dapl_llist_remove_entry(&hca_ptr->ib_trans.list, 
					(DAPL_LLIST_ENTRY*)&cr->entry);
		dapl_os_free( cr, sizeof(*cr) );
		continue;
	    }
	          
	    FD_SET( cr->l_socket, &rfds ); /* add to select set */
	    if ( cr->l_socket > max_fd )
		max_fd = cr->l_socket;

	    /* individual select poll to check for work */
	    FD_ZERO(&rfd);
	    FD_SET(cr->l_socket, &rfd);
	    dapl_os_unlock(&hca_ptr->ib_trans.lock);	
	    to.tv_sec  = 0;
	    to.tv_usec = 0;
	    if ( select(cr->l_socket + 1,&rfd, NULL, NULL, &to) < 0) {
		dapl_dbg_log (DAPL_DBG_TYPE_CM,
			  " thread: ERR %s on cr %p sk %d\n", 
			  strerror(errno), cr, cr->l_socket);
		close(cr->l_socket);
		cr->l_socket = -1;
	    } else if ( FD_ISSET(cr->l_socket, &rfd) && 
			dapli_socket_accept(cr)) {
		close(cr->l_socket);
		cr->l_socket = -1;
	    }
	    dapl_os_lock( &hca_ptr->ib_trans.lock );
	    next_cr =  dapl_llist_next_entry(&hca_ptr->ib_trans.list,
					     (DAPL_LLIST_ENTRY*)&cr->entry );
	} 
	dapl_os_unlock( &hca_ptr->ib_trans.lock );
	to.tv_sec  = 0;
	to.tv_usec = 100000; /* wakeup and check destroy */
	select(max_fd + 1, &rfds, NULL, NULL, &to);
	dapl_os_lock( &hca_ptr->ib_trans.lock );
    } 
    dapl_os_unlock( &hca_ptr->ib_trans.lock );	
    hca_ptr->ib_trans.cr_state = IB_THREAD_EXIT;
    dapl_dbg_log(DAPL_DBG_TYPE_UTIL," cr_thread(hca %p) exit\n",hca_ptr);
}

/*
 * Local variables:
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 8
 * End:
 */
